<?php
session_start();

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    echo "<div class='text-center py-4 text-red-500'>Access denied. Please login as admin.</div>";
    exit;
}

// Simple database connection
$servername = "localhost";
$username = "root";
$password = "";
$dbname = "reviewprcom_db";

try {
    $conn = new mysqli($servername, $username, $password, $dbname);
    if ($conn->connect_error) {
        throw new Exception("Connection failed: " . $conn->connect_error);
    }
} catch (Exception $e) {
    echo "<div class='text-center py-4 text-red-500'>Database connection failed: " . $e->getMessage() . "</div>";
    exit;
}

// Check if order ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    header("Location: admin_dashboard.php");
    exit;
}

$order_id = intval($_GET['id']);

// Get order from database
$sql = "SELECT * FROM orders WHERE id = ?";
$stmt = $conn->prepare($sql);
if (!$stmt) {
    echo "<div class='text-center py-4 text-red-500'>Database error: " . $conn->error . "</div>";
    exit;
}

$stmt->bind_param("i", $order_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows == 0) {
    echo "<div class='text-center py-4 text-red-500'>Order not found.</div>";
    exit;
}

$order = $result->fetch_assoc();
$stmt->close();

// Create a mock product for display since we may not have product_id
$product = [
    'name' => $order['product_name'],
    'description' => 'Professional review service',
    'price' => $order['price'],
    'image' => 'data:image/svg+xml;charset=UTF-8,%3Csvg width="80" height="80" viewBox="0 0 80 80" fill="none" xmlns="http://www.w3.org/2000/svg"%3E%3Crect width="80" height="80" fill="%23667eea"/%3E%3Cpath d="M40 20L50 35H30L40 20Z" fill="white"/%3E%3C/svg%3E'
];

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order Details - <?php echo htmlspecialchars($order['order_id']); ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f7fa;
            margin: 0;
            padding: 0;
        }

        .admin-header {
            background: white;
            padding: 1rem 2rem;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .admin-header h1 {
            margin: 0;
            color: #333;
        }

        .admin-nav {
            display: flex;
            gap: 1rem;
        }

        .btn-nav {
            padding: 0.5rem 1rem;
            background: #667eea;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            transition: background 0.3s ease;
        }

        .btn-nav:hover {
            background: #5a67d8;
        }

        .btn-back {
            background: #6c757d;
        }

        .btn-back:hover {
            background: #545b62;
        }

        .order-container {
            max-width: 1000px;
            margin: 2rem auto;
            padding: 0 2rem;
        }

        .order-details-card {
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
            margin-bottom: 2rem;
        }

        .order-header {
            background: #667eea;
            color: white;
            padding: 1.5rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .order-header h2 {
            margin: 0;
        }

        .status-badge {
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.9rem;
            text-transform: uppercase;
            font-weight: bold;
        }

        .status-pending { background: #fff3cd; color: #856404; }
        .status-processing { background: #cce7ff; color: #0066cc; }
        .status-completed { background: #d4edda; color: #155724; }
        .status-cancelled { background: #f8d7da; color: #721c24; }

        .order-content {
            padding: 2rem;
        }

        .order-info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 2rem;
            margin-bottom: 2rem;
        }

        .info-section {
            background: #f8f9fa;
            padding: 1.5rem;
            border-radius: 8px;
        }

        .info-section h3 {
            margin: 0 0 1rem 0;
            color: #333;
            border-bottom: 2px solid #667eea;
            padding-bottom: 0.5rem;
        }

        .info-item {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
        }

        .info-label {
            font-weight: 600;
            color: #666;
        }

        .info-value {
            color: #333;
        }

        .product-section {
            margin-top: 2rem;
        }

        .product-item {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            background: #f8f9fa;
            border-radius: 8px;
            margin-bottom: 1rem;
        }

        .product-image {
            width: 80px;
            height: 80px;
            border-radius: 8px;
            object-fit: cover;
        }

        .product-details h4 {
            margin: 0 0 0.5rem 0;
            color: #333;
        }

        .product-price {
            color: #e74c3c;
            font-weight: bold;
        }

        .order-actions {
            padding: 1.5rem;
            background: #f8f9fa;
            display: flex;
            gap: 1rem;
            justify-content: flex-end;
        }

        .btn-action {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 1rem;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-pending {
            background: #ffc107;
            color: #212529;
        }

        .btn-pending:hover {
            background: #e0a800;
        }

        .btn-processing {
            background: #17a2b8;
            color: white;
        }

        .btn-processing:hover {
            background: #138496;
        }

        .btn-completed {
            background: #28a745;
            color: white;
        }

        .btn-completed:hover {
            background: #218838;
        }

        .btn-cancelled {
            background: #dc3545;
            color: white;
        }

        .btn-cancelled:hover {
            background: #c82333;
        }

        @media (max-width: 768px) {
            .admin-header {
                flex-direction: column;
                gap: 1rem;
            }

            .order-container {
                padding: 0 1rem;
            }

            .order-header {
                flex-direction: column;
                gap: 1rem;
                text-align: center;
            }

            .info-section {
                margin-bottom: 1rem;
            }

            .order-actions {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <header class="admin-header">
        <h1><i class="fas fa-file-invoice"></i> Order Details</h1>
        <nav class="admin-nav">
            <a href="admin_dashboard.php" class="btn-nav btn-back">
                <i class="fas fa-arrow-left"></i> Back to Dashboard
            </a>
            <a href="manage-products.php" class="btn-nav">
                <i class="fas fa-box"></i> Products
            </a>
            <a href="admin_logout.php" class="btn-nav" style="background: #e53e3e;">
                <i class="fas fa-sign-out-alt"></i> Logout
            </a>
        </nav>
    </header>

    <div class="order-container">
        <div class="order-details-card">
            <div class="order-header">
                <div>
                    <h2>Order #<?php echo htmlspecialchars($order['order_id']); ?></h2>
                    <p style="margin: 0; opacity: 0.9;">
                        <i class="fas fa-calendar"></i>
                        <?php echo date('F j, Y \a\t g:i A', strtotime($order['order_date'])); ?>
                    </p>
                </div>
                <div style="text-align: right;">
                    <span class="status-badge status-<?php echo $order['status']; ?>">
                        <?php echo ucfirst($order['status']); ?>
                    </span>
                    <p style="margin: 0.5rem 0 0 0; font-size: 1.2rem; font-weight: bold;">
                        $<?php echo number_format($order['total'], 2); ?>
                    </p>
                </div>
            </div>

            <div class="order-content">
                <div class="order-info-grid">
                    <div class="info-section">
                        <h3><i class="fas fa-user"></i> Customer Information</h3>
                        <div class="info-item">
                            <span class="info-label">Email:</span>
                            <span class="info-value"><?php echo htmlspecialchars($order['customer_email']); ?></span>
                        </div>
                        <?php if (!empty($order['customer_name'])): ?>
                        <div class="info-item">
                            <span class="info-label">Name:</span>
                            <span class="info-value"><?php echo htmlspecialchars($order['customer_name']); ?></span>
                        </div>
                        <?php endif; ?>
                        <?php if (!empty($order['promo_code'])): ?>
                            <div class="info-item">
                                <span class="info-label">Promo Code:</span>
                                <span class="info-value"><?php echo htmlspecialchars($order['promo_code']); ?></span>
                            </div>
                        <?php endif; ?>
                    </div>

                    <div class="info-section">
                        <h3><i class="fas fa-shopping-bag"></i> Order Details</h3>
                        <?php if (!empty($order['reviews_per_day'])): ?>
                        <div class="info-item">
                            <span class="info-label">Reviews per day:</span>
                            <span class="info-value"><?php echo $order['reviews_per_day']; ?> reviews</span>
                        </div>
                        <?php endif; ?>
                        <div class="info-item">
                            <span class="info-label">Quantity:</span>
                            <span class="info-value"><?php echo $order['quantity'] ?? 1; ?></span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Price per review:</span>
                            <span class="info-value">$<?php echo number_format($order['price'], 2); ?></span>
                        </div>
                    </div>
                </div>

                <?php if (!empty($order['review_location'])): ?>
                    <div class="info-section" style="width: 100%;">
                        <h3><i class="fas fa-map-marker-alt"></i> Review Location</h3>
                        <p style="margin: 0; padding: 0.5rem; background: white; border-radius: 4px; border-left: 4px solid #667eea;">
                            <?php echo htmlspecialchars($order['review_location']); ?>
                        </p>
                    </div>
                <?php endif; ?>

                <?php if (!empty($order['customer_notes'])): ?>
                    <div class="info-section" style="width: 100%;">
                        <h3><i class="fas fa-sticky-note"></i> Special Instructions</h3>
                        <p style="margin: 0; padding: 0.5rem; background: white; border-radius: 4px; border-left: 4px solid #28a745;">
                            <?php echo htmlspecialchars($order['customer_notes']); ?>
                        </p>
                    </div>
                <?php endif; ?>

                <?php if (!empty($order['file_name']) || !empty($order['file_link'])): ?>
                    <div class="info-section" style="width: 100%;">
                        <h3><i class="fas fa-paperclip"></i> Attachments</h3>
                        <?php if (!empty($order['file_name'])): ?>
                            <p style="margin: 0.25rem 0; padding: 0.5rem; background: white; border-radius: 4px;">
                                <i class="fas fa-file"></i> <?php echo htmlspecialchars($order['file_name']); ?>
                            </p>
                        <?php endif; ?>
                        <?php if (!empty($order['file_link'])): ?>
                            <p style="margin: 0.25rem 0; padding: 0.5rem; background: white; border-radius: 4px;">
                                <i class="fas fa-link"></i>
                                <a href="<?php echo htmlspecialchars($order['file_link']); ?>" target="_blank">
                                    <?php echo htmlspecialchars($order['file_link']); ?>
                                </a>
                            </p>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <div class="product-section">
                    <h3><i class="fas fa-box"></i> Product Information</h3>
                    <div class="product-item">
                        <img src="<?php echo htmlspecialchars($product['image']); ?>" alt="<?php echo htmlspecialchars($product['name']); ?>" class="product-image">
                        <div class="product-details">
                            <h4><?php echo htmlspecialchars($product['name']); ?></h4>
                            <p style="margin: 0; color: #666;"><?php echo htmlspecialchars($product['description']); ?></p>
                            <div class="product-price">$<?php echo number_format($product['price'], 2); ?> each</div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="order-actions">
                <?php if ($order['status'] === 'pending'): ?>
                    <button onclick="updateOrderStatus('processing', <?php echo $order['id']; ?>)" class="btn-action btn-processing">
                        <i class="fas fa-play"></i> Start Processing
                    </button>
                <?php elseif ($order['status'] === 'processing'): ?>
                    <button onclick="updateOrderStatus('completed', <?php echo $order['id']; ?>)" class="btn-action btn-completed">
                        <i class="fas fa-check"></i> Mark Complete
                    </button>
                    <button onclick="updateOrderStatus('cancelled', <?php echo $order['id']; ?>)" class="btn-action btn-cancelled">
                        <i class="fas fa-times"></i> Cancel Order
                    </button>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
        function updateOrderStatus(newStatus, orderId) {
            if (confirm(`Are you sure you want to mark this order as ${newStatus}?`)) {
                // Create a form to submit the status update
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = 'update_order_status.php';
                form.innerHTML = `
                    <input type="hidden" name="order_id" value="${orderId}">
                    <input type="hidden" name="status" value="${newStatus}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }
    </script>
</body>
</html>
