<?php
// Enable output compression for faster loading
if (extension_loaded('zlib') && !ob_get_level()) {
    ob_start('ob_gzhandler');
}

include 'db.php';

// Rate limiting
if (!checkRateLimit('products_api', 30, 60)) {
    handleError('Too many requests', 429);
}

try {
    // Get products from database with prepared statement for security
    $stmt = $conn->prepare("SELECT id, name, price, category, image, description, detailed_description, stock, position FROM products WHERE stock > 0 ORDER BY position ASC, id ASC");
    
    if (!$stmt) {
        throw new Exception("Database query failed");
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    $productsArray = array();
    
    while($row = $result->fetch_assoc()) {
        // Sanitize output data
        $productsArray[] = [
            'id' => (int)$row['id'],
            'name' => sanitizeInput($row['name']),
            'price' => (float)$row['price'],
            'category' => sanitizeInput($row['category']),
            'image' => sanitizeInput($row['image']),
            'description' => sanitizeInput($row['description']),
            'detailed_description' => sanitizeInput($row['detailed_description']),
            'stock' => (int)$row['stock'],
            'position' => (int)$row['position']
        ];
    }
    
    $stmt->close();
    
    // Set cache headers for performance
    header('Cache-Control: public, max-age=300'); // 5 minutes cache
    header('ETag: ' . md5(json_encode($productsArray)));
    
    // Return JSON response
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($productsArray, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    
} catch (Exception $e) {
    error_log("Products API error: " . $e->getMessage());
    handleError("Unable to fetch products", 500);
}

$conn->close();
?>
