<?php
session_start();

// Simple database connection
$servername = "localhost";
$username = "root";
$password = "";
$dbname = "reviewprcom_db";

try {
    $conn = new mysqli($servername, $username, $password, $dbname);
    if ($conn->connect_error) {
        die("Connection failed: " . $conn->connect_error);
    }
} catch (Exception $e) {
    die("Database connection failed. Please run the admin_login.php first to set up the database.");
}

// Check if user is logged in as admin
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Get order counts for stats
$order_counts = [
    'total' => 0,
    'pending' => 0,
    'processing' => 0,
    'completed' => 0,
    'cancelled' => 0
];

// Check if orders table exists, if not create it with the correct structure
$conn->query("CREATE TABLE IF NOT EXISTS orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_id VARCHAR(20) NOT NULL,
    product_id INT,
    product_name VARCHAR(255) NOT NULL,
    price DECIMAL(10,2) NOT NULL,
    quantity INT DEFAULT 1,
    total DECIMAL(10,2) NOT NULL,
    customer_email VARCHAR(100) NOT NULL,
    review_location TEXT,
    promo_code VARCHAR(50),
    reviews_per_day INT,
    file_name VARCHAR(255),
    file_link TEXT,
    customer_notes TEXT,
    status ENUM('pending','processing','completed','cancelled') DEFAULT 'pending',
    order_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP
)");

// Check if products table exists, if not create it with sample data
$products_check = $conn->query("SHOW TABLES LIKE 'products'");
if ($products_check->num_rows == 0) {
    $conn->query("CREATE TABLE products (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        description TEXT,
        price DECIMAL(10,2) NOT NULL,
        stock INT DEFAULT 100,
        category VARCHAR(100),
        position INT DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )");
    
    // Insert sample products that match your UI
    $sample_products = [
        ['Google Reviews Package', 'Boost your Google Business Profile with authentic, high-quality reviews from real customers. Our comprehensive service includes review generation, management, and response strategies to improve your online reputation and local SEO rankings.', 49.99, 100, 'reviews', 1],
        ['Facebook Reviews Boost', 'Enhance your Facebook business page with genuine customer reviews that build trust and credibility. Our service focuses on creating meaningful interactions and positive feedback that resonates with your target audience.', 39.99, 100, 'reviews', 2],
        ['Trustpilot Professional', 'Build consumer trust with verified Trustpilot reviews that showcase your business reliability and customer satisfaction. Our premium service ensures compliance with Trustpilot strict verification process.', 59.99, 100, 'reviews', 3],
        ['Yelp Reputation Management', 'Improve your Yelp business profile with authentic reviews and professional reputation management. Our service helps you navigate Yelp complex algorithm while building genuine customer relationships.', 54.99, 100, 'reviews', 4]
    ];
    
    foreach ($sample_products as $product) {
        $stmt = $conn->prepare("INSERT INTO products (name, description, price, stock, category, position) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("ssdiis", $product[0], $product[1], $product[2], $product[3], $product[4], $product[5]);
        $stmt->execute();
    }
}

// Get actual order counts
try {
    $result = $conn->query("SELECT status, COUNT(*) as count FROM orders GROUP BY status");
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $order_counts[$row['status']] = $row['count'];
            $order_counts['total'] += $row['count'];
        }
    }
    
    // If no orders exist, get total count
    if ($order_counts['total'] == 0) {
        $total_result = $conn->query("SELECT COUNT(*) as total FROM orders");
        if ($total_result) {
            $order_counts['total'] = $total_result->fetch_assoc()['total'];
        }
    }
} catch (Exception $e) {
    error_log("Error getting order counts: " . $e->getMessage());
}

// Get recent orders
$recent_orders = [];
try {
    $stmt = $conn->prepare("SELECT * FROM orders ORDER BY order_date DESC LIMIT 10");
    if ($stmt) {
        $stmt->execute();
        $recent_orders = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }
} catch (Exception $e) {
    error_log("Error getting recent orders: " . $e->getMessage());
}

// Get product statistics
$product_stats = [];
$product_result = $conn->query("SELECT COUNT(*) as total_products, SUM(stock) as total_stock FROM products");
if ($product_result) {
    $product_stats = $product_result->fetch_assoc();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - ReviewPR</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .sidebar {
            min-height: calc(100vh - 4rem);
        }
        .active-tab {
            background-color: #3b82f6;
            color: white;
        }
        .active-tab:hover {
            background-color: #2563eb;
        }
    </style>
</head>
<body class="bg-gray-100">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <div class="bg-gray-800 text-white w-64 flex-shrink-0 sidebar">
            <div class="p-4 border-b border-gray-700">
                <h1 class="text-2xl font-bold">ReviewPR Admin</h1>
            </div>
            <nav class="mt-4">
                <a href="#dashboard" class="block py-3 px-4 text-gray-300 hover:bg-gray-700 active-tab">
                    <i class="fas fa-tachometer-alt mr-2"></i> Dashboard
                </a>
                <a href="#orders" class="block py-3 px-4 text-gray-300 hover:bg-gray-700">
                    <i class="fas fa-shopping-cart mr-2"></i> Orders
                </a>
                <a href="#products" class="block py-3 px-4 text-gray-300 hover:bg-gray-700">
                    <i class="fas fa-box mr-2"></i> Products
                </a>
                <a href="#customers" class="block py-3 px-4 text-gray-300 hover:bg-gray-700">
                    <i class="fas fa-users mr-2"></i> Customers
                </a>
                <a href="#analytics" class="block py-3 px-4 text-gray-300 hover:bg-gray-700">
                    <i class="fas fa-chart-line mr-2"></i> Analytics
                </a>
                <a href="admin_logout.php" class="block py-3 px-4 text-gray-300 hover:bg-gray-700 mt-auto">
                    <i class="fas fa-sign-out-alt mr-2"></i> Logout
                </a>
            </nav>
        </div>

        <!-- Main Content -->
        <div class="flex-1 overflow-auto">
            <!-- Top Navigation -->
            <header class="bg-white shadow-sm">
                <div class="flex justify-between items-center p-4">
                    <h2 class="text-xl font-semibold text-gray-800">Dashboard</h2>
                    <div class="flex items-center space-x-4">
                        <a href="index.php" target="_blank" class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors">
                            <i class="fas fa-external-link-alt mr-2"></i>View Website
                        </a>
                        <span class="text-gray-600">Welcome, Admin</span>
                        <div class="w-8 h-8 rounded-full bg-blue-500 flex items-center justify-center text-white">
                            <i class="fas fa-user"></i>
                        </div>
                    </div>
                </div>
            </header>

            <!-- Dashboard Content -->
            <main class="p-6">
                <!-- Stats Cards -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                    <div class="bg-white p-6 rounded-lg shadow">
                        <div class="flex items-center">
                            <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                                <i class="fas fa-shopping-cart text-xl"></i>
                            </div>
                            <div class="ml-4">
                                <p class="text-gray-500">Total Orders</p>
                                <h3 class="text-2xl font-bold"><?php echo number_format($order_counts['total']); ?></h3>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white p-6 rounded-lg shadow">
                        <div class="flex items-center">
                            <div class="p-3 rounded-full bg-green-100 text-green-600">
                                <i class="fas fa-check-circle text-xl"></i>
                            </div>
                            <div class="ml-4">
                                <p class="text-gray-500">Completed</p>
                                <h3 class="text-2xl font-bold"><?php echo number_format($order_counts['completed']); ?></h3>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white p-6 rounded-lg shadow">
                        <div class="flex items-center">
                            <div class="p-3 rounded-full bg-yellow-100 text-yellow-600">
                                <i class="fas fa-clock text-xl"></i>
                            </div>
                            <div class="ml-4">
                                <p class="text-gray-500">Pending</p>
                                <h3 class="text-2xl font-bold"><?php echo number_format($order_counts['pending']); ?></h3>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white p-6 rounded-lg shadow">
                        <div class="flex items-center">
                            <div class="p-3 rounded-full bg-red-100 text-red-600">
                                <i class="fas fa-times-circle text-xl"></i>
                            </div>
                            <div class="ml-4">
                                <p class="text-gray-500">Cancelled</p>
                                <h3 class="text-2xl font-bold"><?php echo number_format($order_counts['cancelled']); ?></h3>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                    <div class="bg-white p-6 rounded-lg shadow">
                        <h3 class="text-lg font-medium text-gray-900 mb-4">Products</h3>
                        <p class="text-gray-600 mb-4">Total Products: <?php echo $product_stats['total_products'] ?? 0; ?></p>
                        <p class="text-gray-600 mb-4">Total Stock: <?php echo $product_stats['total_stock'] ?? 0; ?></p>
                        <a href="manage-products.php" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors">
                            <i class="fas fa-box mr-2"></i>Manage Products
                        </a>
                    </div>
                    <div class="bg-white p-6 rounded-lg shadow">
                        <h3 class="text-lg font-medium text-gray-900 mb-4">Website</h3>
                        <p class="text-gray-600 mb-4">View your live website with the Read More buttons</p>
                        <a href="index.php" target="_blank" class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors">
                            <i class="fas fa-external-link-alt mr-2"></i>View Website
                        </a>
                    </div>
                    <div class="bg-white p-6 rounded-lg shadow">
                        <h3 class="text-lg font-medium text-gray-900 mb-4">Orders</h3>
                        <p class="text-gray-600 mb-4">Manage customer orders and track status</p>
                        <button onclick="alert('Order management coming soon!')" class="bg-purple-600 text-white px-4 py-2 rounded-lg hover:bg-purple-700 transition-colors">
                            <i class="fas fa-shopping-cart mr-2"></i>View All Orders
                        </button>
                    </div>
                </div>

                <!-- Recent Orders -->
                <div class="bg-white rounded-lg shadow overflow-hidden">
                    <div class="p-6 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Recent Orders</h3>
                    </div>
                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Order Details</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Customer Info</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Service Details</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status & Actions</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php if (count($recent_orders) > 0): ?>
                                    <?php foreach ($recent_orders as $order): ?>
                                        <tr class="hover:bg-gray-50">
                                            <!-- Order Details -->
                                            <td class="px-6 py-4">
                                                <div class="flex flex-col">
                                                    <div class="text-sm font-medium text-gray-900">
                                                        <i class="fas fa-hashtag mr-1"></i><?php echo htmlspecialchars($order['order_id']); ?>
                                                    </div>
                                                    <div class="text-xs text-gray-500 mt-1">
                                                        <i class="fas fa-calendar mr-1"></i><?php echo date('M d, Y H:i', strtotime($order['order_date'])); ?>
                                                    </div>
                                                    <div class="text-xs text-green-600 font-medium mt-1">
                                                        <i class="fas fa-dollar-sign mr-1"></i><?php echo number_format($order['total'], 2); ?>
                                                    </div>
                                                </div>
                                            </td>
                                            
                                            <!-- Customer Info -->
                                            <td class="px-6 py-4">
                                                <div class="flex flex-col">
                                                    <div class="text-sm font-medium text-gray-900">
                                                        <i class="fas fa-envelope mr-1"></i><?php echo htmlspecialchars($order['customer_email']); ?>
                                                    </div>
                                                    <?php if (!empty($order['customer_name'])): ?>
                                                    <div class="text-xs text-gray-500 mt-1">
                                                        <i class="fas fa-user mr-1"></i><?php echo htmlspecialchars($order['customer_name']); ?>
                                                    </div>
                                                    <?php endif; ?>
                                                    <?php if (!empty($order['review_location'])): ?>
                                                    <div class="text-xs text-blue-600 mt-1">
                                                        <i class="fas fa-map-marker-alt mr-1"></i><?php echo htmlspecialchars($order['review_location']); ?>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                            
                                            <!-- Service Details -->
                                            <td class="px-6 py-4">
                                                <div class="flex flex-col">
                                                    <div class="text-sm font-medium text-gray-900">
                                                        <?php echo htmlspecialchars($order['product_name']); ?>
                                                    </div>
                                                    <div class="text-xs text-gray-500 mt-1">
                                                        <i class="fas fa-shopping-cart mr-1"></i>Qty: <?php echo $order['quantity'] ?? 1; ?>
                                                    </div>
                                                    <?php if (!empty($order['reviews_per_day'])): ?>
                                                    <div class="text-xs text-purple-600 mt-1">
                                                        <i class="fas fa-star mr-1"></i><?php echo $order['reviews_per_day']; ?> reviews/day
                                                    </div>
                                                    <?php endif; ?>
                                                    <?php if (!empty($order['customer_notes'])): ?>
                                                    <div class="text-xs text-gray-400 mt-1 truncate max-w-xs" title="<?php echo htmlspecialchars($order['customer_notes']); ?>">
                                                        <i class="fas fa-comment mr-1"></i><?php echo htmlspecialchars(substr($order['customer_notes'], 0, 30)) . (strlen($order['customer_notes']) > 30 ? '...' : ''); ?>
                                                    </div>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                            
                                            <!-- Status & Actions -->
                                            <td class="px-6 py-4">
                                                <div class="flex flex-col space-y-2">
                                                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full w-fit
                                                        <?php 
                                                            $statusClass = [
                                                                'pending' => 'bg-yellow-100 text-yellow-800',
                                                                'processing' => 'bg-blue-100 text-blue-800',
                                                                'completed' => 'bg-green-100 text-green-800',
                                                                'cancelled' => 'bg-red-100 text-red-800'
                                                            ][$order['status']] ?? 'bg-gray-100 text-gray-800';
                                                            echo $statusClass;
                                                        ?>">
                                                        <?php echo ucfirst(htmlspecialchars($order['status'])); ?>
                                                    </span>
                                                    
                                                    <div class="flex space-x-2">
                                                        <button onclick="viewOrderDetails(<?php echo $order['id']; ?>)" 
                                                                class="text-xs bg-blue-100 text-blue-700 px-2 py-1 rounded hover:bg-blue-200">
                                                            <i class="fas fa-eye mr-1"></i>View
                                                        </button>
                                                        
                                                        <?php if ($order['status'] === 'pending'): ?>
                                                        <button onclick="updateOrderStatus(<?php echo $order['id']; ?>, 'processing')" 
                                                                class="text-xs bg-green-100 text-green-700 px-2 py-1 rounded hover:bg-green-200">
                                                            <i class="fas fa-play mr-1"></i>Process
                                                        </button>
                                                        <?php elseif ($order['status'] === 'processing'): ?>
                                                        <button onclick="updateOrderStatus(<?php echo $order['id']; ?>, 'completed')" 
                                                                class="text-xs bg-purple-100 text-purple-700 px-2 py-1 rounded hover:bg-purple-200">
                                                            <i class="fas fa-check mr-1"></i>Complete
                                                        </button>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="4" class="px-6 py-12 text-center">
                                            <div class="text-gray-400">
                                                <i class="fas fa-shopping-cart text-4xl mb-4"></i>
                                                <p class="text-lg font-medium">No orders found</p>
                                                <p class="text-sm">Orders will appear here when customers place them</p>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script>
        function updateOrderStatus(orderId, status) {
            const statusText = status.charAt(0).toUpperCase() + status.slice(1);
            if (confirm(`Are you sure you want to mark this order as ${statusText}?`)) {
                // Show loading state
                const button = event.target;
                const originalText = button.innerHTML;
                button.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i>Updating...';
                button.disabled = true;
                
                fetch('update_order_status.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `order_id=${orderId}&status=${status}`
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showNotification('Order status updated successfully!', 'success');
                        setTimeout(() => location.reload(), 1000);
                    } else {
                        showNotification('Error: ' + (data.message || 'Failed to update order status'), 'error');
                        button.innerHTML = originalText;
                        button.disabled = false;
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showNotification('Network error occurred. Please try again.', 'error');
                    button.innerHTML = originalText;
                    button.disabled = false;
                });
            }
        }
        
        function viewOrderDetails(orderId) {
            // Create modal for order details
            const modal = document.createElement('div');
            modal.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50';
            modal.innerHTML = `
                <div class="bg-white rounded-lg p-6 max-w-2xl w-full mx-4 max-h-96 overflow-y-auto">
                    <div class="flex justify-between items-center mb-4">
                        <h3 class="text-lg font-semibold">Order Details</h3>
                        <button onclick="this.closest('.fixed').remove()" class="text-gray-400 hover:text-gray-600">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <div id="order-details-content">
                        <div class="text-center py-4">
                            <i class="fas fa-spinner fa-spin text-2xl text-blue-500"></i>
                            <p class="mt-2">Loading order details...</p>
                        </div>
                    </div>
                </div>
            `;
            document.body.appendChild(modal);
            
            // Fetch order details
            fetch(`order_details.php?id=${orderId}`)
                .then(response => response.text())
                .then(html => {
                    document.getElementById('order-details-content').innerHTML = html;
                })
                .catch(error => {
                    document.getElementById('order-details-content').innerHTML = `
                        <div class="text-center py-4 text-red-500">
                            <i class="fas fa-exclamation-triangle text-2xl"></i>
                            <p class="mt-2">Failed to load order details</p>
                        </div>
                    `;
                });
        }
        
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            const colors = {
                success: 'bg-green-500',
                error: 'bg-red-500',
                info: 'bg-blue-500',
                warning: 'bg-yellow-500'
            };
            
            notification.className = `fixed top-4 right-4 ${colors[type]} text-white px-6 py-3 rounded-lg shadow-lg z-50 transform transition-all duration-300`;
            notification.innerHTML = `
                <div class="flex items-center">
                    <i class="fas fa-${type === 'success' ? 'check' : type === 'error' ? 'times' : 'info'}-circle mr-2"></i>
                    <span>${message}</span>
                </div>
            `;
            
            document.body.appendChild(notification);
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => notification.remove(), 300);
            }, 3000);
        }

        // Tab switching functionality
        document.querySelectorAll('nav a').forEach(link => {
            link.addEventListener('click', function(e) {
                // Remove active class from all links
                document.querySelectorAll('nav a').forEach(a => a.classList.remove('active-tab'));
                // Add active class to clicked link
                this.classList.add('active-tab');
                
                // Here you would typically load the corresponding content via AJAX
                // For now, we'll just prevent the default link behavior
                e.preventDefault();
                const section = this.getAttribute('href').substring(1);
                console.log('Loading section:', section);
            });
        });
    </script>
</body>
</html>