// Modern Service Worker for ReviewPR.com
// Version: 3.0.0 - Optimized for Modern UI
// Updated: 2025-10-13

const CACHE_NAME = 'reviewpr-modern-v3.0.0';
const STATIC_CACHE = 'reviewpr-static-v3.0.0';
const DYNAMIC_CACHE = 'reviewpr-dynamic-v3.0.0';

// Files to cache immediately
const STATIC_FILES = [
    './',
    './index.php',
    './products.php',
    './checkout.php',
    './login.php',
    './assets/css/optimized.min.css',
    './assets/js/modern-script.js',
    './site.webmanifest',
    './robots.txt'
];

// External resources to cache
const EXTERNAL_RESOURCES = [
    'https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css',
    'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css',
    'https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap'
];

// Dynamic cache patterns
const CACHE_PATTERNS = [
    /^https:\/\/cdn\.jsdelivr\.net\//,
    /^https:\/\/cdnjs\.cloudflare\.com\//,
    /^https:\/\/fonts\.googleapis\.com\//,
    /^https:\/\/fonts\.gstatic\.com\//,
    /\.(?:png|jpg|jpeg|svg|gif|webp|ico)$/,
    /\.(?:css|js)$/
];

// Install event - cache static files
self.addEventListener('install', event => {
    console.log('SW: Installing Modern Service Worker v3.0.0...');
    
    event.waitUntil(
        Promise.all([
            caches.open(STATIC_CACHE).then(cache => {
                console.log('SW: Caching static files');
                return cache.addAll(STATIC_FILES);
            }),
            caches.open(DYNAMIC_CACHE).then(cache => {
                console.log('SW: Caching external resources');
                return cache.addAll(EXTERNAL_RESOURCES);
            })
        ])
        .then(() => {
            console.log('SW: Installation complete');
            return self.skipWaiting();
        })
        .catch(error => {
            console.error('SW: Installation failed', error);
        })
    );
});

// Activate event - clean up old caches
self.addEventListener('activate', event => {
    console.log('SW: Activating...');
    
    event.waitUntil(
        caches.keys()
            .then(cacheNames => {
                return Promise.all(
                    cacheNames.map(cacheName => {
                        if (cacheName !== STATIC_CACHE && cacheName !== DYNAMIC_CACHE) {
                            console.log('SW: Deleting old cache', cacheName);
                            return caches.delete(cacheName);
                        }
                    })
                );
            })
            .then(() => {
                console.log('SW: Activated successfully');
                return self.clients.claim();
            })
    );
});

// Fetch event - optimized caching strategy
self.addEventListener('fetch', event => {
    const { request } = event;
    
    // Skip non-GET requests and non-HTTP requests
    if (request.method !== 'GET' || !request.url.startsWith('http')) {
        return;
    }
    
    // Different strategies for different types of requests
    if (request.destination === 'document') {
        // Network first for HTML documents
        event.respondWith(networkFirstStrategy(request));
    } else if (shouldCache(request.url)) {
        // Cache first for static assets
        event.respondWith(cacheFirstStrategy(request));
    } else {
        // Network only for API calls and dynamic content
        event.respondWith(fetch(request));
    }
});

// Network first strategy (for HTML documents)
async function networkFirstStrategy(request) {
    try {
        const networkResponse = await fetch(request);
        const cache = await caches.open(DYNAMIC_CACHE);
        cache.put(request, networkResponse.clone());
        return networkResponse;
    } catch (error) {
        console.log('SW: Network failed, trying cache', request.url);
        const cachedResponse = await caches.match(request);
        return cachedResponse || createOfflineResponse();
    }
}

// Cache first strategy (for static assets)
async function cacheFirstStrategy(request) {
    const cachedResponse = await caches.match(request);
    
    if (cachedResponse) {
        // Update cache in background
        updateCacheInBackground(request);
        return cachedResponse;
    }
    
    try {
        const networkResponse = await fetch(request);
        const cache = await caches.open(DYNAMIC_CACHE);
        cache.put(request, networkResponse.clone());
        return networkResponse;
    } catch (error) {
        console.error('SW: Failed to fetch resource', request.url);
        return createFallbackResponse(request);
    }
}

// Update cache in background
function updateCacheInBackground(request) {
    fetch(request)
        .then(response => {
            if (response.ok) {
                caches.open(DYNAMIC_CACHE)
                    .then(cache => cache.put(request, response));
            }
        })
        .catch(() => {
            // Silently fail background updates
        });
}

// Helper function to determine if a URL should be cached
function shouldCache(url) {
    return CACHE_PATTERNS.some(pattern => pattern.test(url));
}

// Create offline response
function createOfflineResponse() {
    return new Response(`
        <!DOCTYPE html>
        <html>
        <head>
            <title>ReviewPR - Offline</title>
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <style>
                body { font-family: Inter, sans-serif; text-align: center; padding: 50px; background: #f9fafb; }
                .offline-container { max-width: 400px; margin: 0 auto; }
                .offline-icon { font-size: 4rem; color: #667eea; margin-bottom: 1rem; }
                h1 { color: #1f2937; margin-bottom: 1rem; }
                p { color: #6b7280; margin-bottom: 2rem; }
                .retry-btn { background: #667eea; color: white; padding: 12px 24px; border: none; border-radius: 8px; cursor: pointer; }
            </style>
        </head>
        <body>
            <div class="offline-container">
                <div class="offline-icon">📡</div>
                <h1>You're Offline</h1>
                <p>Please check your internet connection and try again.</p>
                <button class="retry-btn" onclick="location.reload()">Retry</button>
            </div>
        </body>
        </html>
    `, {
        headers: { 'Content-Type': 'text/html' }
    });
}

// Create fallback response for failed resources
function createFallbackResponse(request) {
    if (request.destination === 'image') {
        return new Response(
            `<svg xmlns="http://www.w3.org/2000/svg" width="300" height="200" viewBox="0 0 300 200">
                <rect width="300" height="200" fill="#f3f4f6"/>
                <text x="150" y="100" text-anchor="middle" dy=".3em" fill="#9ca3af" font-family="Inter, sans-serif">
                    Image unavailable
                </text>
            </svg>`,
            { headers: { 'Content-Type': 'image/svg+xml' } }
        );
    }
    
    return new Response('Resource unavailable', { 
        status: 503,
        statusText: 'Service Unavailable' 
    });
}

console.log('SW: Modern Service Worker v3.0.0 loaded successfully');
